using System;
using System.Data;
using System.Reflection;
using System.Text.RegularExpressions;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Luke Meyer</Developer>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>8/23/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Class which represents the characteristics of a Division</summary>

	#endregion

	public class Division : DivisionBase
	{	
		/// <summary>
		/// Indicates whether the VAMC division has enabled electronic crossmatching.
		/// </summary>
		private bool _electronicCrossmatchIndicator = false;
		/// <summary>
		/// All facilities within the VA are registered with ICCBBA
		/// </summary>
		private string _iccbbaRegistrationNumber;
		///	<summary>
		///	BloodProductModifications keeps track of component modifications.
		///	</summary>
		private BOL.BloodProductModifications _bloodProductModifications;
		/// <summary>
		/// Whether to enable the ABO/Rh interpretations value for automated instruments
		/// </summary>
		private bool _automatedInstrumentIndicator;
		/// <summary>
		/// DivisionAlerts has all of the information related to alerts for this division
		/// </summary>
		private BOL.DivisionAlerts _divisionAlerts;
		/// <summary>
		/// Antibody screen presentation value
		/// </summary>
		private Common.PatientTestingConfiguration _patientTestingConfiguration = Common.PatientTestingConfiguration.Unknown;
		/// <summary>
		/// Maximum number of days a specimenhas before it expires
		/// </summary>
		private short _maximumSpecimenExpirationDays;
		/// <summary>
		/// Medical Director Name
		/// </summary>
		private string _medicalDirectorName;
		/// <summary>
		/// Address
		/// </summary>
		private BOL.Address _address;
		/// <summary>
		/// ReactivityPhaseCode
		/// </summary>
		private Common.ReactivityPhase _reactivityPhaseCode = Common.ReactivityPhase.Unknown;

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/30/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="754"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>Initialized Division object with properties set</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="767"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException raised</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="8157"> 
		///		<ExpectedInput>Null division code</ExpectedInput>
		///		<ExpectedOutput>Argument Null Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overloaded division constructor takes division code as a parameter.
		/// All properties are pre-loaded for given division code.
		/// </summary>
		/// <param name="divisionCode">Division Code</param>
		public Division( string divisionCode ) : base()
		{
			if( divisionCode == null )
				throw( new ArgumentNullException( "divisionCode" ) );

			_bloodProductModifications = new BloodProductModifications( divisionCode );
			
			LoadFromDataRow( DAL.Division.GetDivision( divisionCode ) );
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/30/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4048"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>Division</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="4049"> 
		///		<ExpectedInput>Invalid datarow</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8158"> 
		///		<ExpectedInput>Null datarow</ExpectedInput>
		///		<ExpectedOutput>Argument Null Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overloaded division constructor takes data row as a parameter.
		/// All properties are pre-loaded with data from given row.
		/// </summary>
		/// <param name="dataRow">Data Row</param>
		public Division( DataRow dataRow )
		{
			this.LoadFromDataRow( dataRow );
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1431"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2234"> 
		///		<ExpectedInput>Invalid data row</ExpectedInput>
		///		<ExpectedOutput>InvalidCastException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		public override DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			return dtRow;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/30/2002</CreationDate>
		/// <summary>
		/// Method to load properties of an existing division object with data from a given data row
		/// Implements BR_9.02, BR_9.03, BR_9.05
		/// </summary>
		/// <param name="dtRow">Division Data Row</param>
		protected override void LoadFromDataRow( DataRow dtRow )
		{
			if( dtRow == null )
				throw( new ArgumentNullException( "dtRow" ) );

			base.LoadFromDataRow( dtRow );

			if( this.FacilityGuid != Guid.Empty ) // Loaded by parent class. 
				this._address = new Address( this.FacilityGuid );

			// BR_9.03
			if( dtRow.IsNull(Common.VbecsTables.VamcDivision.ElectronicCrossmatchIndicator ) )
				this.ElectronicCrossmatchIndicator = false;
			else
				this.ElectronicCrossmatchIndicator = (bool)dtRow[ Common.VbecsTables.VamcDivision.ElectronicCrossmatchIndicator ];

			if (dtRow.IsNull(Common.VbecsTables.VamcDivision.IccbbaRegistrationNumber))
				this.IccbbaRegistrationNumber = string.Empty;
			else
				this.IccbbaRegistrationNumber = dtRow[Common.VbecsTables.VamcDivision.IccbbaRegistrationNumber].ToString();
			
			if (dtRow.IsNull(Common.VbecsTables.VamcDivision.AntibodyScreenDisplayCode))
				this.PatientTestingConfiguration = Common.PatientTestingConfiguration.AntibodyOnly;
			else
				this.PatientTestingConfiguration = (Common.PatientTestingConfiguration)dtRow[Common.VbecsTables.VamcDivision.AntibodyScreenDisplayCode];

			if (dtRow.IsNull(Common.VbecsTables.VamcDivision.AutomatedInstrumentIndicator))
				this.AutomatedInstrumentIndicator = false;
			else
				this.AutomatedInstrumentIndicator = (bool)dtRow[Common.VbecsTables.VamcDivision.AutomatedInstrumentIndicator];
	
			if (dtRow.IsNull(Common.VbecsTables.VamcDivision.MaximumSpecimenExpirationDays))
				this.MaximumSpecimenExpirationDays = Convert.ToInt16(Common.SpecimenExpirationDays.Default);
			else
				this.MaximumSpecimenExpirationDays = (short)dtRow[Common.VbecsTables.VamcDivision.MaximumSpecimenExpirationDays];

			if (dtRow.IsNull(Common.VbecsTables.VamcDivision.MedicalDirectorName))
				this.MedicalDirectorName = string.Empty;
			else
				this.MedicalDirectorName = dtRow[Common.VbecsTables.VamcDivision.MedicalDirectorName].ToString();

			if (dtRow.IsNull(Common.VbecsTables.VamcDivision.ReactivityPhaseCode))
				this.ReactivityPhaseCode = Common.ReactivityPhase.Unknown;
			else
				this.ReactivityPhaseCode = Common.Utility.GetReactivityPhaseCodeFromChar(Convert.ToChar(dtRow[Common.VbecsTables.VamcDivision.ReactivityPhaseCode]));

			if( !dtRow.IsNull(Common.VbecsTables.VamcDivision.RowVersion ) )
				this.RowVersion = (byte[])dtRow[ Common.VbecsTables.VamcDivision.RowVersion ];

			this._divisionAlerts = new BOL.DivisionAlerts( dtRow );

			this.IsDirty = false;
			this.IsNew = false;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/30/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="758"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Success</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2242"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save division to file
		/// </summary>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns>Boolean success indicator</returns>
		public bool Save( Common.UpdateFunction lastUpdateFunctionId )
		{
			return DAL.Division.UpdateDivision(
				this.DivisionCode,
				this.ElectronicCrossmatchIndicator,
				this.IccbbaRegistrationNumber,
				Convert.ToInt32(this.PatientTestingConfiguration),
				this.AutomatedInstrumentIndicator,
				this.MedicalDirectorName,

				// Division Alerts
				this.DivisionAlerts.IconAlertsIndicator,
				this.DivisionAlerts.IconOrderAlertType,
				this.DivisionAlerts.IconOrdersNeededHours,
				this.DivisionAlerts.IconRefreshRate,
				this.DivisionAlerts.PrintAlertsIndicator,
				this.DivisionAlerts.PrinterName,
				this.DivisionAlerts.PrintOrderAlertType,
				this.DivisionAlerts.PrintOrderNeededHours,
				this.DivisionAlerts.LoginMessage,
				this.MaximumSpecimenExpirationDays,
				this.RowVersion,
				lastUpdateFunctionId,
				//CR 2717
				this.ReactivityPhaseCode);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6927"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6928"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves medical director names for all divisions.
		/// </summary>
		/// <returns>Array of medical director names.</returns>
		public static string[] GetMedicalDirectors()
		{
			DataTable _dt = DAL.Division.GetMedicalDirectorNames( Common.LogonUser.LogonUserDivisionCode );
			string[] _directors = new string[ _dt.Rows.Count ];
			int nCount = 0;
			nCount = _dt.Rows.Count;
			for( int i = 0; i < nCount; i++ )
				_directors[i] = _dt.Rows[i][ Common.VbecsTables.VamcDivision.MedicalDirectorName ].ToString();

			return _directors;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/23/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="284"> 
		///		<ExpectedInput>Electronic crossmatch indicator</ExpectedInput>
		///		<ExpectedOutput>Updated division object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="763"> 
		///		<ExpectedInput>Electronic crossmatch indicator = true for Transfusion-only facility</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether the VAMC division has enabled electronic crossmatching.
		/// Implements: BR_9.06
		/// </summary>
		public bool ElectronicCrossmatchIndicator
		{
			get 
			{	
				return this._electronicCrossmatchIndicator;
			}
			set 
			{	
				//BR_9.06
				if ((!this.ServiceTypeIndicator) && (value == true))
				{
					throw new BOL.BusinessObjectException("Electronic Crossmatch Indicator cannot be set for transfusion-only sites");
				}

				if( value == this._electronicCrossmatchIndicator )
					return;

				this._electronicCrossmatchIndicator = value;				
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/30/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="745"> 
		///		<ExpectedInput>Division object</ExpectedInput>
		///		<ExpectedOutput>ICCBBA registration number of division</ExpectedOutput>
		///	</Case>
		///<Case type="0" testid ="765"> 
		///		<ExpectedInput>Valid ICCBBA registration number</ExpectedInput>
		///		<ExpectedOutput>Updated division object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="766"> 
		///		<ExpectedInput>Invalid ICCBBA registration number</ExpectedInput>
		///		<ExpectedOutput>BusinessLayerException raised</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets ICCBBA Registration Number of VA facility
		/// Implements BR_9.04, BR_9.10
		/// </summary>
		public string IccbbaRegistrationNumber
		{
			get
			{
				return this._iccbbaRegistrationNumber;
			}
			set
			{
				Regex iccbbaRegex = Common.RegularExpressions.IccbbaRegistrationNumber();
				
				//BR_9.04
				if ((value.Trim().Length > 0) && (!iccbbaRegex.IsMatch(value)))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
						Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true ),
						value ).ResString;

					throw new BOL.BusinessObjectException(errorMessage);
				}
				
				if (value != this._iccbbaRegistrationNumber)
				{
					this._iccbbaRegistrationNumber = value;
					this.IsDirty = true;
				}
				//BR_9.10
				RuleBroken( MethodBase.GetCurrentMethod().Name, ((this.BloodProductModifications != null) && (this.BloodProductModifications.IsProductModificationEnabled) && (value.Trim().Length != 5)) );
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1419"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>LocalSupplierGuid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7446"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Collection Facility GUID
		/// </summary>
		public Guid LocalSupplierGuid
		{
			get
			{
				return VolatileLocalSupplierGuid;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1418"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>BloodProductModifications object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2282"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Product modifications for a division
		/// Implements BR_9.10
		/// </summary>
		public BOL.BloodProductModifications BloodProductModifications
		{
			get
			{
				return _bloodProductModifications;
			}
			set
			{
				_bloodProductModifications = value;
				//
				//BR_9.10
				RuleBroken( "IccbbaRegistrationNumber", ((value != null) && (value.IsProductModificationEnabled) && (this.IccbbaRegistrationNumber.Trim().Length != 5)) );
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2598"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="13"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Whether to allow automated instrument interpretations for division
		/// </summary>
		public bool AutomatedInstrumentIndicator
		{
			get
			{
				return _automatedInstrumentIndicator;
			}
			set
			{
				_automatedInstrumentIndicator = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/8/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2603"> 
		///		<ExpectedInput>Valid Patient Testing Configuration</ExpectedInput>
		///		<ExpectedOutput>Valid Patient Testing Configuration</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="14"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Antibody screen presentation value
		/// </summary>
		public Common.PatientTestingConfiguration PatientTestingConfiguration
		{
			get
			{
				return _patientTestingConfiguration;
			}
			set
			{
				RuleBroken( MethodBase.GetCurrentMethod().Name, value == Common.PatientTestingConfiguration.Unknown );
				_patientTestingConfiguration = value;

				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/9/2003</CreationDate>
		///
		///<TestCases>
		///
		///<Case type="0" testid ="2604"> 
		///		<ExpectedInput>Valid short</ExpectedInput>
		///		<ExpectedOutput>Valid short</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2609"> 
		///		<ExpectedInput>Invalid value</ExpectedInput>
		///		<ExpectedOutput>Additional broken rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Maximum specimen expiration days
		/// Implements BR_9.05
		/// </summary>
		public short MaximumSpecimenExpirationDays
		{
			get
			{
				return _maximumSpecimenExpirationDays;
			}
			set
			{
				RuleBroken( MethodBase.GetCurrentMethod().Name, 
					( ( value < Convert.ToInt16( Common.SpecimenExpirationDays.Minimum ) ) || ( value > Convert.ToInt16( Common.SpecimenExpirationDays.Maximum ) ) ) );
				
				_maximumSpecimenExpirationDays = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3644"> 
		///		<ExpectedInput>DivisionAlerts</ExpectedInput>
		///		<ExpectedOutput>DivisionAlerts</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3645"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>Division alerts.</summary>
		public DivisionAlerts DivisionAlerts
		{
			get
			{
				return _divisionAlerts;
			}
			set
			{
				_divisionAlerts = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3792"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3793"> 
		///		<ExpectedInput>31 char string</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets medical director name
		/// </summary>
		public string MedicalDirectorName
		{
			get
			{
				return _medicalDirectorName;
			}
			set
			{
				if (value.Length > 30)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(
						Common.Utility.ParseStringwithSpaces( MethodBase.GetCurrentMethod().Name, true ), "30" ).ResString;

					throw new BOL.BusinessObjectException( errorMessage );
				}

				this._medicalDirectorName = value;
				this.IsDirty = true;

				RuleBroken( MethodBase.GetCurrentMethod().Name, ( value.Trim().Length < 3 ) ) ;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6929"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6930"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Division address
		/// </summary>
		public BOL.Address Address
		{
			get
			{
				return this._address;
			}
		}
		
		/// <summary>
		/// ReactivityPhaseCode 
		/// CR 2717
		/// </summary>
		public Common.ReactivityPhase ReactivityPhaseCode
		{
			get
			{
				return this._reactivityPhaseCode;
			}
			set
			{
				this._reactivityPhaseCode = value;
				this.IsDirty = true;
			}
		}
	}
}